/*
 * This file contains the code for initializing the WIN32 dynamic link
 * library so that it works with various Windows REXX interpreters.
 */
#ifdef WIN32
#include <windows.h>
#include "rxproto.h"


/* ============================ NON-SHARED DATA ============================= */

/* Where we store the addresses of functions we call within the REXX Interpreter DLL */
RxVariablePoolPtr *RexxVariablePoolA;
RxRegisterFunctionDllPtr *RexxRegisterFunctionDllA;
RxDeregisterFunctionPtr *RexxDeregisterFunctionA;
RxFreeMemoryPtr *REXXFREEMEMORY;
RxAllocateMemoryPtr *REXXALLOCATEMEMORY;


/* ============================== SHARED DATA ==============================
 * NOTE: I specify this data section to be Shared (ie, each running rexx script shares
 * these variables, rather than getting its own copies of these variables). This is
 * because, since I have only globals that are read-only or whose value is the same
 * for all processes, I don't need a separate copy of these for each process that uses
 * this DLL. In Visual C++'s Linker settings, I add "/section:Shared,rws"
 */

#pragma data_seg("Shared")

/* The names of the functions in the REXX Interpreter DLL we call */
const char	RxVariablePool[] = "RexxVariablePool";
const char	RxRegisterFunctionDll[] = "RexxRegisterFunctionDll";
const char	RxDeregisterFunction[] = "RexxDeregisterFunction";
const char	RxFreeMemory[] = "RexxFreeMemory";
const char	RxAllocateMemory[] = "RexxAllocateMemory";

/* The names of some REXX Interpreter's we know about. (They must be in DLL form) */
const char	Interpreters[] = {"Reginald\0\
Regina\0\
Rexx\0\
\0"};	// Must end with a 0

#pragma data_seg()




// For REXX interpreters that don't have RexxFreeMemory()
APIRET APIENTRY dummyFree(PVOID mem)
{
	GlobalFree(mem);
	return(0);
}




// For REXX interpreters that don't have RexxAllocateMemory()
PVOID APIENTRY dummyAlloc(ULONG bytes)
{
	return(GlobalAlloc(GMEM_FIXED, bytes));
}





/****************************** DLLMain() *****************************
 * Automatically called by Win32 when the DLL is loaded or unloaded.
 */

BOOL WINAPI DllMain(HANDLE hinstDLL, DWORD fdwReason, LPVOID lpvReserved)
{
	switch (fdwReason)
	{
		/* ============================================================== */
		case DLL_PROCESS_ATTACH:
		{
			HMODULE			handle;
			const char		*ptr;

			/* Find out which REXX Interpreter loaded me */
			ptr = &Interpreters[0];
			do
			{
				/* If this is it, stop searching */
				if ((handle = GetModuleHandle(ptr))) goto good;

				/* Next interpreter name */
				while (*(ptr)++);

				/* Any more? */
			} while (*ptr);

			/* I don't know about any others, so fail */
			return(0);

			/* Get the address of the RexxRegisterFunctionDll() function */
good:		if (!(RexxRegisterFunctionDllA = (RxRegisterFunctionDllPtr *)GetProcAddress((HINSTANCE)handle, &RxRegisterFunctionDll[0]))

				/* Get the address of the RexxVariablePool() function */
				|| !(RexxVariablePoolA = (RxVariablePoolPtr *)GetProcAddress((HINSTANCE)handle, &RxVariablePool[0]))

				/* Get the address of the RexxDeregisterFunction() function */
				|| !(RexxDeregisterFunctionA = (RxDeregisterFunctionPtr *)GetProcAddress((HINSTANCE)handle, &RxDeregisterFunction[0])))
			{
				return(0);
			}

			/* Get the address of RexxAllocateMemory/RexxFreeMemory, or use generic
			 * functions in this DLL
			 */
			if (!(REXXFREEMEMORY = (RxFreeMemoryPtr *)GetProcAddress((HINSTANCE)handle, &RxFreeMemory[0])))
			{
				REXXFREEMEMORY = dummyFree;
				REXXALLOCATEMEMORY = dummyAlloc;
			}
			else
				REXXALLOCATEMEMORY = (RxAllocateMemoryPtr *)GetProcAddress((HINSTANCE)handle, &RxAllocateMemory[0]);

			break;
		}

		/* ============================================================== */
		case DLL_THREAD_ATTACH:
		{
			/* We don't need to do anything for THREAD ATTACH, so we can
			 * disable this support.
			 */
			DisableThreadLibraryCalls(hinstDLL);
			break;
		}
/*
		case DLL_THREAD_DETACH:
			break;

		case DLL_PROCESS_DETACH:
		{
		}
*/
	}

	/* Success */
	return(1);
}

#endif