 /******************* CgiReadPost() *****************
  * Reads the input data from some FORM on the web
  * page. The CGI interface must have queued that data
  * for us using the POST method.
  *
  * error = CgiReadPost(Variable)
  *
  * Variable = The name of the REXX variable where
  *				the data should be returned. Do not
  *				quote this name.
  *
  * RETURNS: An empty string if success, or an error
  * message if failure.
  *
  * NOTES:
  * CgiReadPost() returns the input data if the
  * REQUEST_METHOD is "POST". If the method is not
  * POST, then Variable is DROP'ed. Variable is also
  * DROP'ed if the REQUEST_METHOD is "POST" but there
  * is no input in the standard input.
  *
  * Note: The returned data does NOT have plus
  * signs (+) converted to spaces or hex ASCII %XX
  * encodings converted to characters.  
  */

CgiReadPost:
	USE ARG	input

	/* Make sure that the CGI interface has used the POST method
	 * to send us the data. It has set the environment variable
	 * named 'REQUEST_METHOD' to a string that tells what
	 * method was used (ie, 'GET' or 'POST'). We can query that
	 * environment variable's value using GETENV().
	 */
	IF GETENV('REQUEST_METHOD') \== "POST" THEN DROP input
	ELSE DO
		/* Note: Reginald triggers NOVALUE if we use GETENV() to read
		 * an environment variable, and no such variable has been set.
		 * So let's do this to catch error below.
		 */
		SIGNAL ON NOVALUE NAME CgiReadPostError

		/* Get the length of the data. The CGI interface has set the
		 * environment variable named 'CONTENT_LENGTH' to the number
		 * of characters in the data.
		 */
		input = GETENV('CONTENT_LENGTH')

		/* The CGI interface has written the data to standard input. We
		 * can retrieve it using CHARIN() and omitting the filename
		 * (ie, first) argument.
		 */
		SIGNAL ON NOTREADY NAME CgiReadPostError2
		input = CHARIN(,, input)
	END

	/* Success */
	RETURN ""

CgiReadPostError:
	input = CONDITION('D') "environment variable was not set!"
	/* Raise an ERROR condition in the caller */
	RAISE ERROR 1 DESCRIPTION (input) RETURN (input)

CgiReadPostError2:
	DROP input
	/* Raise an ERROR condition in the caller */
	RAISE ERROR 2 DESCRIPTION ("Form data could not be read!") RETURN ("Form data could not be read!")