/* calc.rex -- a Classic Rexx desk calculator
 this program can be executed by any Classic Rexx interpreter

 Keywords
  Numerical calculator, Generic REXX expression calculator,
  INTERPRET example, REXX expression experimentation

 Usage
  roo calc any_REXX_expression

 Arguments
  A REXX expression

 Files used
  Standard output

 Exit codes
   0   => expression performed
 non-0 => usage error

 Input record format
  N/A

 Sample input file
  N/A

 Sample output
  13 ** 47 is: 2.26640520E+52

 Example of use
  [1] roo calc 13 ** 47 ---> 2.26640520E+52
  [2] roo calc dirs('\*')
  [3] roo calc files('*.rex')
  [4] roo calc words('To be, or not to be')

 Explanation
  This procedure is used to compute the value of any REXX
  expression.

  In the examples above,
   [1] Calculates 13 raised to an exponent of 47
   [2] Shows all subdirectories of the root directory
   [3] Shows files matching pattern "*.rex" within the current directory
   [4] Shows the number of words in the quotation

   As simple as this procedure looks it is a highly capable calculator!
*/

if arg(1) = "" | arg(1) = "?" then signal usagemsg

parse arg inargs                      /* acquire requested REXX expression */

if verify( inargs, '()', 'Match' ) > 0 then
  inargs = inargs || optional_paren( inargs )

interpret 'say inargs "is:"' inargs        /* what you get is what you say */

exit 0

usagemsg:
    say "Usage"
    say " roo calc any_REXX_expression"
    say ""
    say " Examples of use"
    say "  [1] roo calc 13 ** 47 ---> 2.26640523E+52"
    say "  [2] roo calc dirs('\*')"
    say "  [3] roo calc files('*.rex')"
    say "  [4] roo calc words('To be, or not to be')"
    exit 1

optional_paren : procedure
  arg inargs
  count = 0
  do while inargs <> ''
    p = verify( inargs, '()', 'Match' )
    if p = 0 then leave
    count = count + word( '-1 1', 1 + (substr( inargs, p, 1 ) = ')' ) )
    inargs = substr( inargs, p+1 )
    end
  return substr( ')', 1, count < 0 )

